<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

add_action( 'plugins_loaded', 'wc_gateway_eraty_init', 0 );

function wc_gateway_eraty_init()
{

    if (!class_exists('WC_Payment_Gateway')) {
        return;
    }

    class WC_Gateway_Eraty extends WC_Payment_Gateway
    {
        public function __construct()
        {
            $this->id = 'eraty';
            $this->api_url = ERATY__FORM_URL;
            $this->icon = ERATY__PLUGIN_URL . '/images/buy.png';
            $this->has_fields = true;

            $this->init_form_fields();
            $this->init_settings();

            $this->title = $this->get_option('title') ?: 'eRaty Santander Consumer Bank';

            add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
            add_action( 'woocommerce_api_wc_gateway_' . $this->id, array( $this, 'check_eraty_response' ) );
            add_action( 'woocommerce_receipt_' . $this->id, array( $this, 'eraty_pay_for_order') );
        }

        public function payment_fields()
        {
            global $woocommerce;
            $total = $woocommerce->cart->total;

            ?>
            <p>Kup na raty z Santander Consumer Bank</p>
            <p>Przed złożeniem wniosku możesz zapoznać się z procedurą udzielenia kredytu ratalnego oraz obliczyć raty.</p>
            <p class="eraty-confirmation-buttons">
                <?php render_terms_button(); ?>
                <?php render_evaluate_button($total); ?>
            </p>
            <p class="eraty-confirmation-credit-rule">
                <input type="checkbox" id="<?php echo esc_attr( $this->id ) . '-confirmation' ?>" name="<?php echo esc_attr( $this->id ) . '-confirmation' ?>"/>
                Zapoznałem się <a href="#" class="eraty-terms-button">z procedurą udzielenia kredytu konsumenckiego na zakup towarów i usług eRaty Santander Consumer Bank</u></a>
            </p>
            <p>Aby rozpocząć proces składania wniosku ratalnego należy zapoznać się z procedurą udzielenia kredytu, a następnie kliknąć na przycisk składania zamówienia.</p>
            <?php
        }

        public function validate_fields()
        {
            if ($this->get_post_value($this->id . '-confirmation') != 'on') {
                wc_add_notice(__('Aby kontynuować proces zakupów na raty, należy potwierdzić fakt zapoznania się z procedurą udzielania kredytu konsumenckiego na zakup towarów i usług eRaty Santander Consumer Bank', 'eraty'), 'error');
                return false;
            }

            return true;
        }

        public function init_form_fields() {
            $this->form_fields = array(
                'enabled' => array(
                    'title' => __( 'Włącz/Wyłącz', 'eraty' ),
                    'type' => 'checkbox',
                    'label' => __( 'Włącz system eRaty', 'eraty' ),
                    'default' => 'yes'
                ),
                'title' => array(
                    'title'   => __( 'Tytuł', 'eraty' ),
                    'type'    => 'text',
                    'default' => 'eRaty Santander Consumer Bank'
                ),
            );
        }

        public function init_settings() {
            parent::init_settings();


            $this->enabled  = (!empty($this->settings['enabled']) && 'yes' === $this->settings['enabled']) ? 'yes' : 'no';
        }

        public function is_available() {
            global $woocommerce;
            return parent::is_available() && isset($woocommerce->cart) && priceIsValidated($woocommerce->cart->total);
        }

        private function get_post_value($key) {
            return isset($_POST[$key]) ? $_POST[$key] : null;
        }

        private function format_price($value) {
            return number_format($value, 2, '.', '');
        }

        public function process_payment($order_id) {

            $order = new WC_Order($order_id);
            $email_new_order = WC()->mailer()->get_emails()['WC_Email_New_Order'];
            $email_new_order->trigger( $order_id );

            return array(
                'result' => 'success',
                'redirect' => $order->get_checkout_payment_url( true ),
            );
        }

        public function eraty_pay_for_order( $order_id ) {

            global $woocommerce;

            $order = new WC_Order($order_id);
            $total = $this->format_price($woocommerce->cart->total);

            $items = $order->get_items();

            $params = array(
                'wartoscTowarow' => $total,
                'liczbaSztukTowarow' => count($items),
                'numerSklepu' => get_option('eraty_shop_id'),
                'typProduktu' => '0',
                'nrZamowieniaSklep' => $order_id,
                'sposobDostarczeniaTowaru' => 'Przesyłka',
                'blokadaWplaty' => 0,

                'pesel' => null,
                'imie' => $order->get_billing_first_name(),
                'nazwisko' => $order->get_billing_last_name(),
                'email' => $order->get_billing_email(),
                'telKontakt' => $order->get_billing_phone(),
                'ulica' => $order->get_billing_address_1(),
                'nrDomu' => $order->get_billing_address_2(),
                'nrMieszkania' => null,
                'miasto' => $order->get_billing_city(),
                'kodPocz' => $order->get_billing_postcode(),

                'char' => 'ISO',
                'wniosekZapisany' => str_replace( 'https:', 'http:', add_query_arg(array(
                        'wc-api' => 'WC_Gateway_Eraty',
                        'status' => 'saved',
                        'order_id' => $order_id,
                        'application_id' => ''
                    ), home_url( '/' ) ) ) . '=',
                'wniosekAnulowany' => str_replace( 'https:', 'http:', add_query_arg(array(
                        'wc-api' => 'WC_Gateway_Eraty',
                        'status' => 'cancelled',
                        'order_id' => $order_id,
                        'application_id' => ''
                    ), home_url( '/' ) ) ) . '=',
            );

            $productsTotal = 0;
            $nr = 0;
            foreach($items as $item) {

                $itemPrice = $this->format_price(wc_get_price_including_tax($item->get_product()));

                if($itemPrice > 0){
                    $nr++;

                    $params['idTowaru' . $nr] = $item->get_product_id();;
                    $params['nazwaTowaru' . $nr] = $item->get_name();
                    $params['liczbaSztukTowaru' . $nr] = $item->get_quantity();
                    $params['wartoscTowaru' . $nr] = $itemPrice;
                    $params['jednostkaTowaru' . $nr] = 'sztuki';

                    $productsTotal += wc_get_price_including_tax($item->get_product());
                }

            }


            if (($shipping = $this->format_price($order->get_shipping_total())) && $shipping > 0)
            {
                if($this->format_price($productsTotal + $shipping) != $total){
                    $shipping = $this->format_price($productsTotal - $total);
                    if($shipping < 1){
                        $shipping = $this->format_price($total - $productsTotal);
                    }
                }
                $nr++;
                $params['idTowaru' . $nr] = 'KosztPrzesylki';
                $params['nazwaTowaru' . $nr] = 'Koszt przesylki';
                $params['liczbaSztukTowaru' . $nr] = 1;
                $params['wartoscTowaru' . $nr] = $shipping;
                $params['jednostkaTowaru' . $nr] = 'sztuki';
            }

            echo '<form id="eratyform" action="' . $this->api_url . '" method="post">';
            foreach ($params as $key => $param){
                echo '<input type="hidden" name="'.$key.'" value="'.$param.'">';
            }
            echo '</form>';
            echo '<script>
                    jQuery("body").prepend("<div id=eraty-redirect>Trwa przekierowanie</div>");
                        jQuery(document).find("#eraty-redirect").css({
                        "backgroundColor": "white",
                        "color": "black",
                        "position": "fixed",
                        "zIndex": "999",
                        "width": "100%",
                        "height": "100%",
                        });
                        jQuery(window).off("beforeunload");
                        jQuery( "#eratyform" ).submit();
            </script>
            ';
        }

        function check_eraty_response() {
            $status = $_GET['status'] ?: 'cancelled';
            $applicationId = $_GET['application_id'];
            if($applicationId ?? null){
                $orderId = $_GET['order_id'];
                $order = wc_get_order( $orderId );
                $order->update_meta_data('_eraty_application_id', $applicationId);
                $order->update_status('on-hold', __( 'Wniosek złożony', 'eraty' ), true);
                $order->save();
            }
            $return_button = '<p><a href="' . esc_url( wc_get_page_permalink( 'shop' ) ) . '" class="button">' . __('Wróć do sklepu', 'eraty') . '</a></p>';

            if ($status == 'saved'){
                wp_die('<b>Dziękujemy za złożenie wniosku. </b>' .
                    '<br><br>Numer wniosku kredytowego/decyzja to: <b>' . $applicationId . '</b>.' .
                    '<br><br><b>W przypadku pozytywnej wstępnej weryfikacji oczekuj na kontakt telefoniczny z konsultantem Santander Consumer Bank</b>' .
                    '<br><br>Podczas rozmowy telefonicznej sporządzi razem z Tobą umowę ratalną.' .
                    '<br><br>Przygotuj: dowód osobisty oraz drugi dokument tożsamości.' .
                    $return_button
                );
            } else {
                wp_die('Zrezygnowałeż z otrzymania kredytu ratalnego.' .
                    '<br>Skontaktuj się z właścicielem sklepu w celu wyjaśnienia sytuacji oraz pomyślnym dokończenia transakcji.' .
                    $return_button
                );
            }
        }
    }

    function add_eraty_gateway_class( $methods ) {
        $methods[] = 'WC_Gateway_Eraty';
        return $methods;
    }

    add_filter( 'woocommerce_payment_gateways', 'add_eraty_gateway_class' );

}
